package org.example;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.universe.mdm.rest.v1.oauth2.type.HttpResponse;
import com.universe.mdm.rest.v1.oauth2.type.OAuth2Provider;
import com.universe.mdm.rest.v1.oauth2.type.OAuth2UserInfo;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Locale;
import java.util.Properties;

public class HydraOAuth2Provider implements OAuth2Provider {

    private static final ObjectMapper objectMapper = new ObjectMapper();

    private static final String PROVIDER_ID = "providerId";
    private static final String CLIENT_ID = "clientId";
    private static final String CLIENT_SECRET = "clientSecret";
    private static final String SCOPE = "scope";
    private static final String AUTH_URI = "authUri";
    private static final String TOKEN_URI = "tokenUri";
    private static final String USER_URI = "userUri";

    private final String providerId;
    private final String clientId;
    private final String clientSecret;
    private final String scope;
    private final String authUri;
    private final String tokenUri;
    private final String userUri;

    public HydraOAuth2Provider() {

        try (InputStream is = getClass().getClassLoader().getResourceAsStream("config.properties")) {

            var properties = new Properties();
            properties.load(is);

            providerId = validate(properties, PROVIDER_ID);
            clientId = validate(properties, CLIENT_ID);
            clientSecret = validate(properties, CLIENT_SECRET);
            scope = validate(properties, SCOPE, false);
            authUri = validate(properties, AUTH_URI);
            tokenUri = validate(properties, TOKEN_URI);
            userUri = validate(properties, USER_URI);

        } catch (FileNotFoundException e) {
            throw new RuntimeException("File config.properties is not found", e);
        } catch (IOException e) {
            throw new RuntimeException("Unknown error reading config.properties", e);
        }
    }

    private String validate(Properties prop, String name) {
        return validate(prop, name, true);
    }

    private String validate(Properties prop, String name, boolean required) {
        String value = prop.getProperty(name);
        if (required) {
            if (value == null || value.isEmpty()) {
                throw new RuntimeException("Missing required property: " + name);
            }
        }

        return value;
    }

    @Override
    public String getId() {
        return providerId;
    }

    @Override
    public String getClientId() {
        return clientId;
    }

    @Override
    public String getClientSecret() {
        return clientSecret;
    }

    @Override
    public String getScope() {
        return scope;
    }

    @Override
    public String getAuthorizationUri() {
        return authUri;
    }

    @Override
    public String getTokenUri() {
        return tokenUri;
    }

    @Override
    public boolean isBasicAuthenticationForTokenRequest() {
        return true;
    }

    @Override
    public String getUserInfoUri() {
        return userUri;
    }

    @Override
    public OAuth2UserInfo extractUserInfo(HttpResponse response) {

        if (response.getStatusCode() != 200) {
            throw new RuntimeException("Error getting user info, http code " + response.getStatusCode());
        }

        try {
            JsonNode json = objectMapper.readTree(response.getBody());

            String name = toText(json, "name");
            if (name == null || name.isEmpty()) {
                throw new RuntimeException("Missing required field: name");
            }

            String login = "hydra_" + providerId + "_" + name.replace(' ', '_');

            String email = toText(json, "email");
            String firstName = toText(json, "given_name");
            String middleName = toText(json, "middle_name");
            String lastName = toText(json, "family_name");
            String localeString = toText(json, "locale");

            Locale locale;
            if (localeString != null && (localeString.equals("ru") || localeString.equals("RU"))) {
                locale = Locale.forLanguageTag("ru");
            } else {
                locale = Locale.ENGLISH;
            }

            var i = new OAuth2UserInfo(login);
            i.setEmail(email);
            i.setFirstName(firstName);
            i.setMiddleName(middleName);
            i.setLastName(lastName);
            i.setLocale(locale);
            i.setActive(true);

            return i;

        } catch (IOException e) {
            throw new RuntimeException("Unknown error reading user info response", e);
        }
    }

    private String toText(JsonNode json, String fieldName) {
        JsonNode node = json.get(fieldName);
        if (node == null || !node.isTextual()) {
            return null;
        }

        return node.asText();
    }
}
